/** @file   GuardianTank.cpp
 * @brief   Implementation of GuardianTank class.
 * @version $Revision: 1.5 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "GuardianTank.h"
#include "Settings.h"
#include "eng2d.h"
#include "www_map.h"
#include "RedrawQueue.h"
#include "WarGlobals.h"
#include "GuardianTankController.h"
#include "GfxManager.h"
#include "SoundSamples.h"
#include "AnimId.h"
#include "AnimPlayer.h"
using namespace eng2d;
using std::string;


namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int GuardianTank::KGunNose1;
const int GuardianTank::KGunNose2;
const int GuardianTank::KReloadCounter;


///
/// Constructors, destructor and operators
/// ============================================================================

GuardianTank::GuardianTank(const string& aRoute) :
  GameObject(),
  iForwardVelocity(1.0),
  iGraphicUid(-1),
  iSoundSource(0)
{
  int oid = ObjectID::TYPE_GUARDIANTANK;
  iForwardVelocity = Settings::floatObjProp(oid, "spd:");
  iGraphicUid = Settings::intObjProp(oid, "gfxUid:");
  this->setArmor(Settings::floatObjProp(oid, "armor:"));
  this->boundingSphere(Settings::floatObjProp(oid, "bsphere:"));
  float attackRange = Settings::floatObjProp(oid, "attack_range:");
  
  Vec2D gunPos1( Settings::floatObjProp(oid, "gun1_nose_x:"),
                 Settings::floatObjProp(oid, "gun1_nose_y:") );
  Vec2D gunPos2( Settings::floatObjProp(oid, "gun2_nose_x:"),
                 Settings::floatObjProp(oid, "gun2_nose_y:") );
  this->addCtrlPoint( gunPos1 );
  this->addCtrlPoint( gunPos2 );
  
  GuardianTankController* controller = new GuardianTankController( this );
  Route route = WarGlobals::pLevel->getRoute( aRoute );
  controller->setRoute( route );
  controller->setAttackRange( attackRange );
  this->setController( controller );
  
  // Create the sound source
  iSoundSource = new SoundSource( "tankmove", Vec2D(0,0) );
  Sound::createSoundSource( iSoundSource );
}


GuardianTank::~GuardianTank()
{
}


void GuardianTank::update()
{
  if ( this->state() != GameObject::STATE_LIVING ) {
    return;
  }
  if ( m_pController != 0 ) {
    m_pController->update();
    if ( m_pController->forward() != 0 ) {
      Vec2D moveVec( 0, -iForwardVelocity );
      moveVec.rotate( this->angle() );
      this->move( moveVec );
    }
    if ( m_pController->shoot() != 0 ) {
      this->shoot();
    }
    this->changeAngle( m_pController->turn() );
  }
  
  if (iSoundSource != 0) {
    iSoundSource->position( this->position() );
    if ( m_pController->forward() != 0 ) {
      iSoundSource->turnOn();
    } else {
      iSoundSource->turnOff();
    }
  }
}


void GuardianTank::redraw( RedrawQueue* aQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  BITMAP* gfx = GfxManager::findBitmap( iGraphicUid );
  if ( gfx != 0 ) {
    Vec2D spritePos = this->position();
    spritePos -= Vec2D( Map::scrollX, Map::scrollY );
    spritePos -= Vec2D( gfx->w/2, gfx->h/2 );
    aQueue->addRotatedSprite( RedrawQueue::PRI_ABOVE_NORMAL,
                              spritePos.intX(), spritePos.intY(),
                              itofix(this->angle()), gfx );
  }
}


void GuardianTank::makeSound( GameObject::SoundID aSoundId ) const
{
  int sampleId = -1;
  switch (aSoundId) {
    case ( GameObject::SND_DIE ): {
      sampleId = SMP_GRENADE;
      break;
    }
    case ( GameObject::SND_PAIN ): {
      sampleId = SMP_METALHIT;
      break;
    }
    case ( GameObject::SND_ATTACK ): {
      sampleId = SMP_SENTRYSHOOT3;
      break;
    }
    default: {
      sampleId = -1;
      break;
    }
  }
  Sound::playSample( sampleId, false );
}


bool GuardianTank::hitByBullet( Bullet* aBullet )
{
  bool ownBullet = aBullet->iOwner == this;
  if ( ownBullet == true ) {
    return false;
  }
  if ( aBullet->iType != Bullet::ESniperRifle ) {
    this->causeDamage( aBullet );
  }
  this->makeSound( GameObject::SND_PAIN );
  ParticleSparks* sparks = new ParticleSparks(aBullet->iPosition, aBullet->velocity(), 12);
  WarGlobals::pPartManager->addSystem( sparks );
  return true;
}


void GuardianTank::kill()
{
  this->hidden( true );
  this->state( GameObject::STATE_KILLED );
  iSoundSource->turnOff();
  
  const Animation& anim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  AnimPlayer::spawn( anim, this->position() );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
  if ( Settings::explosionLights == true ) {
    AnimPlayer::spawn( lightAnim, this->position(), 0 );
  }
  this->makeSound( GameObject::SND_DIE );
}


ObjectID::Type GuardianTank::objectType() const
{
  return ObjectID::TYPE_GUARDIANTANK;
}


bool GuardianTank::reloading() const
{
  if ( this->getCounter( KReloadCounter ) < 0 ) {
    return false;
  }
  return true;
}


void GuardianTank::shoot()
{
  if ( this->reloading() != true ) {
    Vec2D playerPos = WarGlobals::pObjTable->pPlayer->position();
    
    Vec2D gunV1( this->getCtrlPoint(KGunNose1) );
    Vec2D gunV2( this->getCtrlPoint(KGunNose2) );
    gunV1 += this->position();
    gunV2 += this->position();
    Vec2D diffV1 = playerPos - gunV1;
    Vec2D diffV2 = playerPos - gunV2;
    diffV1.norm();
    diffV2.norm();
    
    Bullet* bullet1 = BulletTable::createBullet( this, gunV1, Bullet::ESentryFireball );
    Bullet* bullet2 = BulletTable::createBullet( this, gunV2, Bullet::ESentryFireball );
    diffV1 *= bullet1->velocity().length();
    diffV2 *= bullet2->velocity().length();
    bullet1->setVelocity( diffV1 );
    bullet2->setVelocity( diffV2 );
    
    WarGlobals::pBulletManager->spawnBullet(bullet1);
    WarGlobals::pBulletManager->spawnBullet(bullet2);
    this->makeSound( GameObject::SND_ATTACK );
    const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
    AnimPlayer::spawn( flameAnim, gunV1, 0 );
    AnimPlayer::spawn( flameAnim, gunV2, 0 );
    
    if ( Settings::shootingLights == true ) {
      const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
      AnimPlayer::spawn( lightAnim, gunV1, 0 );
      AnimPlayer::spawn( lightAnim, gunV2, 0 );
    }
    
    int reloadDelay = Settings::intObjProp(ObjectID::TYPE_GUARDIANTANK, "reload:");
    this->setCounter( KReloadCounter, reloadDelay );
  }
}


};  // end of namespace
